<?php
/**
 * Mercadopago API interconnection and processing library for the IPN interface, Prestashop version
 *
 * @author    Rinku Kazeno <development@kazeno.co>
 *
 * @copyright Copyright (c) 2012-2015, Rinku Kazeno
 * @license   This module is licensed to the user, upon purchase
 *  from either Prestashop Addons or directly from the author,
 *  for use on a single commercial Prestashop install, plus an
 *  optional separate non-commercial install (for development/testing
 *  purposes only). This license is non-assignable and non-transferable.
 *  To use in additional Prestashop installations an additional
 *  license of the module must be purchased for each one.
 *
 *  The user may modify the source of this module to suit their
 *  own business needs, as long as no distribution of either the
 *  original module or the user-modified version is made.
 *
 *  @file-version 2.19
 */

/**
 * 
 * Notification Sample:
 *  http://www.yoursite.com/notifications?topic=payment&id=identificador-de-notificación-de-pago
 * 
 * JSON (API 2.0) Transaction Query Response Sample:
    {
        paging: {
            total: 76,
            limit: 2,
            offset: 10
        }
        results: [
            {
                collection:  {
                    id: id-del-pago,
                    site_id: "Identificador de país",
                    date_created: "2011-09-20T00:00:00.000-04:00",
                    date_approved: null,
                    last_modified: "2011-10-19T16:44:34.000-04:00",
                    money_release_date: "2011-10-04T17:32:49.000-04:00",
                    operation_type: "regular_payment",
                    collector_id: id-del-vendedor,
                    sponsor_id: ,
                    payer: {
                        id: id-del-comprador,
                        nickname: "comprador",
                        first_name: "Comprador",
                        last_name: "Comprador",
                        phone: {
                            area_code: "2222",
                            number: "46537890",
                            extension: null
                        },
                        email: "email-del-comprador"
                    },
                    external_reference: "id-referencia-del-vendedor",
                    merchant_order_id: ,
                    reason: "Descripción de lo que se está pagando",
                    transaction_amount: 2,
                    total_paid_amount: 2,
                    currency_id: "Tipo de moneda",
                    shipping_cost: 0,
                    status: "cancelled",
                    status_detail: "expired",
                    released: "yes",
                    payment_type: "ticket",
                }
            },
            {
                collection:  {
                    ...
                }
            }
        ]
    }
 *
 */

class MercadopagoInterconnector
{
    const BASE_URL = 'https://api.mercadolibre.com/';

    protected static function isJson($string)
    {
        return is_object(Tools::jsonDecode($string));
    }

    protected static function baseRequest($subpath, $method, $data='')
    {
        $url = parse_url(self::BASE_URL . $subpath);
        $query = isset($url['query']) ? "?{$url['query']}" : '';
        if ($fp = fsockopen('tls://'.$url['host'], 443)) {
            fwrite($fp, "{$method} {$url['path']}{$query} HTTP/1.0\r\n");
            fwrite($fp, "Host: {$url['host']}\r\n");
            fwrite($fp, "Accept: application/json\r\n");
            if ($data) {
                if (self::isJson($data))
                    fwrite($fp, "Content-type: application/json\r\n");
                else
                    fwrite($fp, "Content-type: application/x-www-form-urlencoded\r\n");
                fwrite($fp, "Content-length: " . Tools::strlen($data) . "\r\n");
                fwrite($fp, "Connection: close\r\n\r\n");
                fwrite($fp, $data);
            } else {
                fwrite($fp, "Connection: close\r\n\r\n");
            }
            $result = '';
            while(!feof($fp)) {
                $result .= fgets($fp);
            }
            fclose($fp);
            $response = explode("\r\n\r\n", trim($result));
            return $response[1];
        } else return FALSE;
    }

    protected static function parseJson($text)
    {
        $json = Tools::substr($text, strpos($text, '{'), strrpos($text, '}')-strpos($text, '{')+1);
        return Tools::jsonDecode($json, TRUE);
    }


    /**
     * @param array $clientData ( 'id', 'secret' )
     * @return string (Access Token)
     */
    public static function retrieveAccessToken(array $clientData)
    {
        $data = http_build_query(array( 'grant_type' => 'client_credentials', 'client_id' => $clientData['id'], 'client_secret' => $clientData['secret'] ));
        if ($body = self::baseRequest('oauth/token', 'POST', $data)) {
            return self::parseJson($body);
        } else return FALSE;
    }

    /**
     * @param string $transaction
     * @param string $token
     * @param bool $queryTransactionId [query by external reference instead of id]
     * @return array ( Transaction Data )
     */
    public static function queryTransaction($transaction, $token, $queryTransactionId=FALSE, $sandbox=FALSE)
    {
        if (!$token)
            return FALSE;
        $searchParam = $queryTransactionId ? 'external_reference' : 'id';
        $prefix = $sandbox ? 'sandbox/' : '';
        $relUrl = "{$prefix}collections/search?access_token={$token}&{$searchParam}={$transaction}";
        if ($body = self::baseRequest($relUrl, 'GET')) {
            try {
                @$parsedJson = self::parseJson($body);
            } catch (Exception $e) {
                return NULL;
            }
            if (count($parsedJson['results'])) {
                $mainCollection = $parsedJson['results'][0]['collection'];
                $summary = array(
                    'id' => array(),
                    'site_id' => $mainCollection['site_id'],
                    'collector_id' => $mainCollection['collector_id'],
                    'external_reference' => $mainCollection['external_reference'],
                    //'merchant_order_id' => $mainCollection['merchant_order_id'],
                    'currency_id' => $mainCollection['currency_id'],
                    'total_paid_amount' => 0,
                    'transaction_amount' => 0,
                    'shipping_cost' => 0,
                    'mercadopago_fee' => 0,
                    'net_received_amount' => 0,
                );
                $statuses = array();
                foreach ($parsedJson['results'] as $res) {
                    $col = $res['collection'];
                    $summary['id'][] = $col['id'];
                    $statuses[] = $col['status'];
                    if ($col['status'] == 'approved') {
                        $summary['transaction_amount'] += $col['transaction_amount'];
                        $summary['total_paid_amount'] += $col['total_paid_amount'];
                        $summary['shipping_cost'] += $col['shipping_cost'];
                        $summary['mercadopago_fee'] += $col['mercadopago_fee'];
                        $summary['net_received_amount'] += $col['net_received_amount'];
                    }
                }
                $statuses = array_unique($statuses);
                if (in_array('pending', $statuses))
                    $summary['status'] = 'pending';
                elseif (in_array('in_process', $statuses))
                    $summary['status'] = 'in_process';
                elseif (in_array('in_mediation', $statuses))
                    $summary['status'] = 'in_mediation';
                elseif (count($statuses) === 1 && in_array('approved', $statuses))
                    $summary['status'] = 'approved';
                else
                    $summary['status'] = array_pop(array_diff($statuses, array('approved')));

                return $summary;
            }
        }
        return NULL;
    }

    /**
     * @param string $token
     * @param array $productData
     * @param bool $sandbox (sandbox mode status)
     * @return string URL
     */
    public static function getPaymentLink($token, array $productData, $sandbox=FALSE){
        if (!$token)
            return FALSE;
        $subpath = "checkout/preferences?access_token={$token}";
        $data = Tools::jsonEncode($productData);
        if ($body = self::baseRequest($subpath, 'POST', $data)) {
            $parsedJson = self::parseJson($body);
            $init = $sandbox ? 'sandbox_init_point' : 'init_point';
            if (!count($parsedJson) || !isset($parsedJson[$init]))
                throw new Exception($body);
            return $parsedJson[$init];
        } else return FALSE;
    }

}


?>