<?php
/**
* Modulo Kueski
*
* @author    Lievant
* @copyright 2020 Lievant
* @license   Commercial use allowed (Non-assignable & non-transferable),
*            can modify source-code but cannot distribute modifications
*            (derivative works).
*/

class Kueski extends PaymentModule
{
    const GATEWAY_NAME = 'KSK';
    public $id_carrier;
    public $gateway = null;

    public function __construct()
    {
        $this->name = 'kueski';
        $this->tab = 'payments_gateways';
        $this->version = '1.0.3';
        $this->author = 'Lievant';
        $this->module_key = '';
        if (version_compare(_PS_VERSION_, '1.7.1.0') >= 0) {
            $this->controllers = array('redirect');
            $this->currencies = true;
            $this->currencies_mode = 'checkbox';
        }

        if (version_compare(_PS_VERSION_, '1.6.0.0') >= 0) {
            $this->bootstrap = true;
        }

        parent::__construct();

        $this->displayName = $this->l('Kueski Pay');
        $this->description = $this->l('Kueski Pay es un método de pago 100%% digital, con el que podrás pagar tu compra a crédito sin tarjeta.');
        if (version_compare(_PS_VERSION_, '1.7.1.0') >= 0) {
            if (!count(Currency::checkPaymentCurrencies($this->id))) {
                $w = $this->trans('No currency has been set for this module.', array(), 'Modules.Checkpayment.Admin');
                $this->warning = $w;
            }
        }

        if (!defined('_PS_VERSION_')) {
            exit;
        }

        if (Module::isInstalled($this->name)) {
            if (!class_exists('KueskiGateway')) {
                include_once(dirname(__FILE__).'/kueski_gateway.php');
            }
            $this->gateway = KueskiGateway::getInstance(self::GATEWAY_NAME, $this->name, $this);
            if ($this->gateway) {
                $this->warning .= $this->gateway->warning;
                $settings = $this->gateway->getSettings();
                $this->displayName = sprintf($this->displayName, $settings['NAME']);
                $this->description = sprintf($this->description, $settings['NAME']);
                $this->limited_countries = array(Tools::strtolower($settings['ISO']));
            }
        } else {
            $data = include(dirname(__FILE__).'/data-mp-countries.php');
            $settings = $data[Tools::strtoupper(self::GATEWAY_NAME)];
            $this->displayName = sprintf($this->displayName, $settings['NAME']);
            $this->description = sprintf($this->description, $settings['NAME']);
            $this->limited_countries = array(Tools::strtolower($settings['ISO']));
        }
        /* For 1.4.3 and less compatibility */
        $update_config = array('PS_OS_CHEQUE',
                            'PS_OS_PAYMENT',
                            'PS_OS_PREPARATION',
                            'PS_OS_SHIPPING',
                            'PS_OS_CANCELED',
                            'PS_OS_REFUND',
                            'PS_OS_ERROR',
                            'PS_OS_OUTOFSTOCK',
                            'PS_OS_BANKWIRE',
                            'PS_OS_PAYPAL',
                            'PS_OS_WS_PAYMENT');

        foreach ($update_config as $u) {
            if (!Configuration::get($u) && defined('_'.$u.'_')) {
                Configuration::updateValue($u, constant('_'.$u.'_'));
            }
        }
    }
    
    public function install()
    {
        $incompatible_found = false;
        if (!function_exists('curl_version')) {
            $this->_errors[] = $this->l('Curl not installed');
            $incompatible_found = true;
        }
        if ($incompatible_found) {
            return false;
        }

        if (!class_exists('KueskiGateway')) {
            include_once(dirname(__FILE__).'/kueski_gateway.php');
        }

        $this->gateway = KueskiGateway::getInstance(self::GATEWAY_NAME, $this->name, $this);
        $db_created = $this->gateway->installDb();

        if (!$db_created) {
            $this->_errors[] = $this->l('Failed to create the table in the Database');
        }

        $is_14 = version_compare(_PS_VERSION_, '1.5.0.0') < 0;
        $is_16 = version_compare(_PS_VERSION_, '1.6.0.0') >= 0;
        $is_17 = version_compare(_PS_VERSION_, '1.7.0.0') >= 0;
        $is_171 = version_compare(_PS_VERSION_, '1.7.1.0') >= 0;
        $result = $db_created && parent::install()
            && $this->registerHook('orderConfirmation')
            && $this->registerHook('payment')
            && $this->registerHook('updateOrderStatus')
            && ($is_14?$this->registerHook('adminOrder'):$this->registerHook('displayAdminOrder'))
            && ($is_14?$this->registerHook('header'):$this->registerHook('displayHeader'))
            && ($is_14?$this->registerHook('backOfficeHeader'):$this->registerHook('displayBackOfficeHeader'))
            
            && ($is_14?$this->registerHook('PDFInvoice'):$this->registerHook('displayPDFInvoice'))
            && ($is_16?$this->registerHook('dashboardZoneOne'):true)
            && ($is_16?$this->registerHook('dashboardData'):true)
            && ($is_17?$this->registerHook('paymentOptions'):true)
            && ($is_171?$this->registerHook('paymentReturn'):true);
        if ($is_171) {
            $result = $result && $this->registerHook('displayProductAdditionalInfo');
        } else {
            if ($is_14) {
                $result = $result && $this->registerHook('productActions');
            } else {
                $result = $result && $this->registerHook('displayProductButtons');
            }
        }
        if (!$result && $db_created) {
            $this->gateway->uninstall();
        }

        if ($result) {
            @chmod(dirname(__FILE__).'/', 0755);
            $cache_id = 'Module::getModuleIdByName_'.pSQL($this->name);
            Cache::clean($cache_id);
        }

        return $result;
    }
    public function uninstall()
    {
        if ($this->gateway) {
            $this->gateway->uninstall();
        }
        return parent::uninstall();
    }
    public function hookDisplayPDFInvoice($params)
    {
        $order_invoice = $params['object'];
        if (!Validate::isLoadedObject($order_invoice) || !isset($order_invoice->id_order)) {
            return;
        }
        return $this->gateway?$this->gateway->hookDisplayPDFInvoice($params):'';
    }
    public function hookPDFInvoice($params)
    {
        return $this->hookDisplayPDFInvoice($params);
    }
    public function hookDisplayProductButtons($params)
    {
        return;
    }
    public function hookDisplayProductAdditionalInfo($params)
    {
        return $this->hookDisplayProductButtons($params);
    }
    public function hookProductActions($params)
    {
        return $this->hookDisplayProductButtons($params);
    }
    
    public function hookDashboardZoneOne($params)
    {
        return $this->display(__FILE__, 'views/templates/hook/dashboard_zone_one.tpl');
    }
    public function hookDashboardData($params)
    {
        $balance = false;
        if ($this->gateway) {
            $balance = $this->gateway->getBalance();
        }
        if ($balance) {
            return array(
                'data_value' => array(
                    $this->name.'_total_amount' => $balance['total_amount'].' '.$balance['currency_id'],
                    $this->name.'_available_balance' => $balance['available_balance'].' '.$balance['currency_id'],
                    $this->name.'_unavailable_balance' => $balance['unavailable_balance'].' '.$balance['currency_id'],
                ),
                'data_trends' => array(
                ),
                'data_list_small' => array(
                ),
                'data_chart' => array(
                ),
            );
        } else {
            return array(
                'data_value' => array(
                    $this->name.'_total_amount' => '-',
                    $this->name.'_available_balance' => '-',
                    $this->name.'_unavailable_balance' => '-',
                ),
                'data_trends' => array(
                ),
                'data_list_small' => array(
                ),
                'data_chart' => array(
                ),
            );
        }
    }
    public function hookDisplayHeader($params)
    {
        return $this->hookHeader($params);
    }
    public function hookHeader($params)
    {
        if (!$this->gateway) {
            return '';
        }
        $result = $this->gateway->cronjob($this->smarty, __FILE__);
        return $result;
    }
    public function hookDisplayBackOfficeHeader($params)
    {
        return $this->hookBackOfficeHeader($params);
    }
    public function hookBackOfficeHeader($params)
    {
        if ($this->gateway && ($result = $this->gateway->hookBackOfficeHeader($params, $this->smarty, __FILE__))) {
            return $result;
        }
        return '';
    }
    public function hookDisplayAdminOrder($params)
    {
        if (!isset($params['id_order']) || !$this->gateway) {
            return '';
        }
        if ($result = $this->gateway->hookDisplayAdminOrder($params, __FILE__)) {
            $this->context->smarty->assign($result);
            return $this->display(__FILE__, 'views/templates/hook/displayAdminOrder.tpl');
        }
        return '';
    }
    public function hookAdminOrder($params)
    {
        return $this->hookDisplayAdminOrder($params);
    }
    public function hookOrderConfirmation($params)
    {
        if (!$this->active || !$this->gateway) {
            return;
        }
        $order = null;
        if (isset($params['objOrder'])) {
            $order = $params['objOrder'];
        } elseif (isset($params['order'])) {
            $order = $params['order'];
        } else {
            return;
        }
        if ($order->module != $this->name) {
            return;
        }
        $this->context->smarty->assign($this->gateway->hookOrderConfirmation($order));
        return $this->display(__FILE__, 'views/templates/hook/hookorderconfirmation.tpl');
    }
    public function getContent()
    {
        if (!$this->gateway) {
            return;
        }
        return $this->gateway->adminPage($this->smarty, __FILE__);
    }
    public function getPathTemplate()
    {
        return $this->_path;
    }
    public function hookPayment($params)
    {
        $config = $this->gateway->getConfig();
        if ($this->gateway && ($result = $this->gateway->paymentButton16($params))) {
            $this->context->smarty->assign($result);
            return $this->display(__FILE__, 'views/templates/hook/mpCart.tpl');
        }
        return '';
    }
    public function hookPaymentOptions($params)
    {
        if (!$this->active) {
            return;
        }
        if ($this->gateway && ($result = $this->gateway->paymentButton17($params, $this->context))) {
            return $result;
        }
        return array();
    }
    public function hookPaymentReturn($params)
    {
        if (!$this->active) {
            return '';
        }
        return '';
    }
    public function hookUpdateOrderStatus($params)
    {
        if ($this->gateway && ($result = $this->gateway->hookUpdateOrderStatus($params))) {
            return $result;
        }
        return '';
    }
    public function lang($str, $specific = false)
    {
        return $this->l($str, $specific);
    }
    public function waitingPayment()
    {
        return $this->l('Waiting payment on Kueski');
    }
}
